package jp.co.hitachi_kokusai.terminal;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.StringTokenizer;

import jp.co.hitachi_kokusai.Version;

import com.docomo_um.io.IOProperties;
import com.docomo_um.lang.Degree;
import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.DeviceException;
import com.docomo_um.module.ExternalStatusException;
import com.docomo_um.module.IllegalSettingException;
import com.docomo_um.module.ModuleException;
import com.docomo_um.module.ModuleFunctions;
import com.docomo_um.module.ModuleListener;
import com.docomo_um.module.ModuleManager;
import com.docomo_um.module.ModuleProperties;
import com.docomo_um.module.connection.ADChannel;
import com.docomo_um.module.connection.ADConverter;
import com.docomo_um.module.connection.ADConverterFunctions;
import com.docomo_um.module.connection.ConnectionException;
import com.docomo_um.module.connection.ConnectionManager;
import com.docomo_um.module.connection.ConnectionProperties;
import com.docomo_um.module.connection.GPIOConnection;
import com.docomo_um.module.connection.GPIOFunctions;
import com.docomo_um.module.connection.I2CConnection;
import com.docomo_um.module.connection.I2CSlave;
import com.docomo_um.module.connection.I2CSpec;
import com.docomo_um.module.connection.SPIFunctions;
import com.docomo_um.module.connection.Serial;
import com.docomo_um.module.connection.SerialConnection;
import com.docomo_um.module.connection.SerialListener;
import com.docomo_um.module.connection.SerialSpec;
import com.docomo_um.module.location.Location;
import com.docomo_um.module.location.LocationException;
import com.docomo_um.module.location.LocationProperties;
import com.docomo_um.module.location.LocationProvider;
import com.docomo_um.module.location.LocationProviderImpl;
import com.docomo_um.module.net.IPSettingInfo;
import com.docomo_um.module.net.MailException;
import com.docomo_um.module.net.ModemController;
import com.docomo_um.module.net.ModemControllerTerminalListener;
import com.docomo_um.module.net.NetProperties;
import com.docomo_um.module.net.PDPTypeInfo;
import com.docomo_um.module.net.PPPSettingInfo;
import com.docomo_um.module.net.PacketControllerImpl;
import com.docomo_um.module.net.PacketSession;
import com.docomo_um.module.net.Session;
import com.docomo_um.module.net.SessionException;
import com.docomo_um.module.net.ShortMessageException;
import com.docomo_um.module.net.ShortMessageFunctions;
import com.docomo_um.module.net.USSDFunctions;
import com.docomo_um.module.net.USSDServiceTerminalListener;
import com.docomo_um.module.net.VoiceControllerImpl;
import com.docomo_um.module.net.VoiceSession;
import com.docomo_um.module.net.ssl.SSLProperties;
import com.docomo_um.module.net.ssl.SecureHandshakeException;
import com.docomo_um.win.LogListener;
import com.docomo_um.win.Logging;
import com.docomo_um.win.SystemProperties;

/**
 * ターミナルI/Fクラス
 *
 */
public class TerminalIF extends Thread {
	Map<String,TerminalCommand> commandMap;
	Map<String,Session> sessionMap;
	List<String> commandList;
	String termPort = "COM8";
	String msg;
	Serial port = null;
	ModuleManager moduleMan;
	ConnectionManager connectionMan;
	VoiceControllerImpl voiceController;
	PacketControllerImpl packetController;

	/**
	 * コンストラクタ
	 */
	public TerminalIF() {
		sessionMap = new HashMap<String,Session>();
		commandMap = new HashMap<String,TerminalCommand>();
		commandList = new ArrayList<String>();
		commandList.add("Quit");
		commandMap.put( "QUIT", null);
		commandList.add("?");
		commandMap.put( "?", new CommandHelp());
		commandMap.put( "LOCALVERSION", new CommandLocalVersion());
		commandList.add("Reset");
		commandMap.put( "RESET", new CommandReset());
		commandList.add("SystemUpdate");
		commandMap.put( "SYSTEMUPDATE", new CommandSystemUpdate());
		commandList.add("SystemUpdateAbort");
		commandMap.put( "SYSTEMUPDATEABORT", new CommandSystemUpdateAbort());
		commandList.add("CancelSystemUpdate");
		commandMap.put( "CANCELSYSTEMUPDATE", new CommandCancelSystemUpdate());
		commandList.add("UpdateStatus");
		commandMap.put( "UPDATESTATUS", new CommandUpdateSystemStatus());
		commandList.add("SignalStatus");
		commandMap.put( "SIGNALSTATUS", new CommandSignalStatus());
		commandList.add("AreaInfo");
		commandMap.put( "AREAINFO", new CommandAreaInfo());
		commandList.add("Regulation");
		commandMap.put( "REGULATION", new CommandRegulation());
		commandList.add("ReceivedPower");
		commandMap.put( "RECEIVEDPOWER", new CommandReceivedPower());

		commandList.add("Send");
		commandMap.put( "SEND", new CommandSend());
		commandList.add("Receive");
		commandMap.put( "RECEIVE", new CommandReceive());
		commandList.add("Disconnect");
		commandMap.put( "DISCONNECT", new CommandDisconnect());
		commandList.add("USSDSend");
		commandMap.put( "USSDSEND", new CommandUSSDSend());
		commandList.add("Guidance");
		commandMap.put( "GUIDANCE", new CommandGuidance());
		commandList.add("SMSSend");
		commandMap.put( "SMSSEND", new CommandSMSSend());

		commandList.add("I2Csend");
		commandMap.put( "I2CSEND", new CommandI2CSend());

		commandList.add("AD");
		commandMap.put( "AD", new CommandAD());

		commandList.add("GPIO");
		commandMap.put( "GPIO", new CommandGPIO());

		commandList.add("SPIStart");
		commandMap.put( "SPISTART", new CommandSPIStart());

		commandList.add("SPIStop");
		commandMap.put( "SPISTOP", new CommandSPIStop());

		commandList.add("UIM");
		commandMap.put( "UIM", new CommandUIM());

		commandList.add("Location");
		commandMap.put( "LOCATION", new CommandLocation());
		commandList.add("ProvideLocation");
		commandMap.put( "PROVIDELOCATION", new CommandProvideLocation());
		commandList.add("LocationStatus");
		commandMap.put( "LOCATIONSTATUS", new CommandLocationStatus());

		commandList.add("ModuleException");
		commandMap.put( "MODULEEXCEPTION", new CommandModuleException());
		commandList.add("CommunicationFailureException");
		commandMap.put( "COMMUNICATIONFAILUREEXCEPTION", new CommandCommunicationFailureException());
		commandList.add("IOException");
		commandMap.put( "IOEXCEPTION", new CommandIOException());
		commandList.add("DeviceException");
		commandMap.put( "DEVICEEXCEPTION", new CommandDeviceException());
		commandList.add("ExternalStatusException");
		commandMap.put( "EXTERNALSTATUSEXCEPTION", new CommandExternalStatusException());
		commandList.add("IllegalSettingException");
		commandMap.put( "ILLEGALSETTINGEXCEPTION", new CommandIllegalSettingException());
		commandList.add("RegulationException");
		commandMap.put( "REGULATIONEXCEPTION", new CommandRegulationException());
		commandList.add("LoadableModuleException");
		commandMap.put( "LOADABLEMODULEEXCEPTION", new CommandLoadableModuleException());
		commandList.add("USSDServiceException");
		commandMap.put( "USSDSERVICEEXCEPTION", new CommandUSSDServiceException());
		commandList.add("SecureHandshakeException");
		commandMap.put( "SECUREHANDSHAKEEXCEPTION", new CommandSecureHandshakeException());
		commandList.add("IllegalCertificateException");
		commandMap.put( "ILLEGALCERTIFICATEEXCEPTION", new CommandIllegalCertificateException());
		commandList.add("SessionException");
		commandMap.put( "SESSIONEXCEPTION", new CommandSessionException());
		commandList.add("mailException");
		commandMap.put( "MAILEXCEPTION", new CommandMailException());
		commandList.add("LocationException");
		commandMap.put( "LOCATIONEXCEPTION", new CommandLocationException());
		commandList.add("TimerException");
		commandMap.put( "TIMEREXCEPTION", new CommandTimerException());
		commandList.add("ConnectionException");
		commandMap.put( "CONNECTIONEXCEPTION", new CommandConnectionException());
		commandList.add("ShortMessageException");
		commandMap.put( "SHORTMESSAGEEXCEPTION", new CommandShortMessageException());

		TerminalSessionListener sessionListener = new TerminalSessionListener();
		try {
			moduleMan = ModuleManager.getInstance();
		} catch (ModuleException e) {
			// TODO 自動生成された catch ブロック
			e.printStackTrace();
		}
		connectionMan = moduleMan.getConnectionManager();
		voiceController = (VoiceControllerImpl) moduleMan.getVoiceController();
		voiceController.setModemControllerTerminalListener(sessionListener);
		packetController = (PacketControllerImpl) moduleMan.getPacketController();
		packetController.setModemControllerTerminalListener(sessionListener);
		USSDFunctions.setTerminalListener(new TerminalUSSDListener());
		termPort = SystemProperties.getInstance().getTerminalPort();
		Logging.getInstance().setLogListener(new LogMessageListener());

		SerialConnection sio = (SerialConnection) connectionMan.getConnection(ConnectionManager.CONNECTION_TYPE_SERIAL);
		List<SerialSpec> specList = sio.getSerialSpec();
		String portName = ConnectionProperties.getInstance().getTerminalPort(termPort);
		for (int i = 0; i < specList.size(); i++) {
			SerialSpec spec = specList.get(i);
			if (spec.getDeviceName().equals(portName)) {
				System.out.println("Terminal I/F open port" + spec.getDeviceName());
				try {
					port = sio.getSerial(spec);
				} catch (ConnectionException e) {
					// TODO 自動生成された catch ブロック
					e.printStackTrace();
				}
				break;
			}
		}
	}
	/**
	 * ターミナルI/Fスレッド
	 */
	@Override
	public void run() {
		Logging.getInstance().start();
		if (port == null) {
			System.out.println("Error Terminal I/F cannot open serial port!");
			return;
		}
		TerminalMessageListener listener = new TerminalMessageListener();
		try {
			port.setCommSpeed(9600);
			port.setEnable(true);
		} catch (ConnectionException e2) {
			e2.printStackTrace();
			return;
		}
		port.setSerialListener(listener);

		println("==============================================");
		println("um03koVM Terminal I/F Start (SDK " + Version.getVersion() + ")");
		println("==============================================");

		while (true) {
			// コマンドライン入力待ち
			msg = listener.read();
			if (msg == null) {
				break;
			}
			// コマンドライン解析
			StringTokenizer token = new StringTokenizer(msg, " ");
			if (token.hasMoreTokens()) {
				String command = token.nextToken();
				List<String> params = new ArrayList<String>();
				while (token.hasMoreTokens()) {
					params.add(token.nextToken());
				}
				command = command.toUpperCase();
				if (commandMap.containsKey(command)) {
					// コマンド実行
					TerminalCommand tc = commandMap.get(command);
					if (tc != null) {
						try {
							if (tc.CommandProc(params) == false) {
								// 終了
								break;
							}
						} catch (IOException e) {
							e.printStackTrace();
						} catch (Exception e) {
							e.printStackTrace();
							try {
								tc.Usage();
							} catch (IOException e1) {
								e1.printStackTrace();
							}
						}
					}
					else {
						break;
					}
				}
				else {
					String err = "コマンド(" + command + ")は見つかりません。\r\n";
					try {
						port.getOutputStream().write(err.getBytes());
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
		}
		println("===============================");
		println("um03koVM Terminal I/F Terminate");
		println("===============================");
		try {
			port.setEnable(false);
		} catch (ConnectionException e) {
			// TODO 自動生成された catch ブロック
			e.printStackTrace();
		}
		Logging.getInstance().stop();
	}
	private class TerminalMessageListener implements SerialListener {
		private String line;
		private byte[] lineBuffer;
		private byte[] escapeBuffer;
		private static final int BUFFERSIZE = 256;
		private int curPos;
		private ArrayList<String> history;
		private static final int MAX_HISTORY = 256;
		TerminalMessageListener() {
			lineBuffer = new byte[BUFFERSIZE];
			escapeBuffer = new byte[2];
			curPos = 0;
			history = new ArrayList<String>();
		}
		/**
		 * 1行読み込み
		 * @return	1行入力文字列
		 */
		public synchronized String read() {
			try {
				wait();
			} catch (InterruptedException e) {
				return null;
			}
			return line;
		}
		/**
		 * 1行書き込み
		 * @param str	1行入力文字列
		 */
		public synchronized void write(String str) {
			line = str;
			notifyAll();
		}
		@Override
		public void onReceived(Serial serial) {
			String command;
			int received;
			InputStream is = serial.getInputStream();
			OutputStream os = serial.getOutputStream();
			try {
				while ((received = is.read()) > 0) {
					switch (received) {
					case 0x1B:	// escape
						is.read(escapeBuffer);
//						System.out.println(String.valueOf(escapeBuffer[0]) + " " + String.valueOf(escapeBuffer[1]));
						if (escapeBuffer[0] == 0x5B) {
							switch (escapeBuffer[1]) {
							case 0x41: //↑
								break;
							case 0x42: //↓
								break;
							case 0x43: //→
								break;
							case 0x44: //←
								break;
							}
						}
						break;
					case 0x08:	// BS
						if (curPos > 0) {
							curPos--;
							os.write(0x08);
							os.write(0x20);
							os.write(0x08);
						}
						break;
					case 0x09:	// TAB
						break;
					case 0x0D:	// CR
						if (curPos > 0) {
							command = new String(lineBuffer, 0, curPos);
							write(command);
							history.add(0, command);
							if (history.size() > MAX_HISTORY) {
								history.remove(history.size() - 1);
							}
							os.write(0x0d);
							os.write(0x0a);
							curPos = 0;
						}
						break;
					default:
						if (received >= 0x20 && received <= 0x7F) {
							if (curPos < BUFFERSIZE) {
								lineBuffer[curPos++] = (byte)received;
								os.write(received);
							}
						}
						break;
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		@Override
		public void onChangedERStatus(Serial serial, boolean er) {}
	}
	/**
	 * ターミナルに文字列を出力します。
	 * @param msg
	 */
	public void print(String msg) {
		synchronized (this) {
			try {
				port.getOutputStream().write(msg.getBytes());
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	/**
	 * ターミナルに文字列を出力し、改行します。
	 * @param msg
	 */
	public void println(String msg) {
		msg += "\r\n";
		synchronized (this) {
			try {
				port.getOutputStream().write(msg.getBytes());
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * ヘルプコマンド
	 */
	private class CommandHelp implements TerminalCommand {
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.isEmpty()) {
				Usage();
			}
			else {
				for (int i = 0; i < params.size(); i++) {
					String key = params.get(i);
					key = key.toUpperCase();
					if (!commandMap.containsKey(key)) {
						continue;
					}
					TerminalCommand tc = commandMap.get(key);
					tc.Usage();
				}
			}
			return true;
		}
		public void Usage() throws IOException{
			String usage = "help\r\n? <commands>\r\n";
			print(usage);
//			for (Iterator<Entry<String, TerminalCommand>> it = commandMap.entrySet().iterator(); it.hasNext();) {
//				Map.Entry<String, TerminalCommand> entry = (Map.Entry<String, TerminalCommand>)it.next();
//				println(entry.getKey());
//			}
			for (int i = 0; i < commandList.size(); i++) {
				println(commandList.get(i));
			}
		}
	}
	/**
	 * HiKE内部バージョンを表示する。
	 */
	private class CommandLocalVersion implements TerminalCommand {
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			print(Version.getLocalVersion() + "\r\n");
			return true;
		}
		@Override
		public void Usage() throws IOException {
		}
	}
	/**
	 * モジュールマネージャにリセット要求を行う。
	 */
	private class CommandReset implements TerminalCommand {
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			ModuleFunctions.requestReset();
			return true;
		}
		@Override
		public void Usage() throws IOException {
			String usage =	"システムのリセットを要求します。\r\n" +
					"reset\r\n";
			print(usage);
		}
	}
	/**
	 * モジュールマネージャにシステムアップデートの要求を行う
	 */
	private class CommandSystemUpdate implements TerminalCommand {
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			ModuleFunctions.requestSystemUpdate();
			return true;
		}
		@Override
		public void Usage() throws IOException {
			String usage =	"システムアップデートの実行を要求します。\r\n" +
							"systemupdate\r\n";
			print(usage);
		}
	}

	/**
	 * モジュールマネージャにシステムアップデートの中断の要求を行う
	 */
	private class CommandSystemUpdateAbort implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			int status = Integer.valueOf(params.get(0));
			switch (status) {
			case ModuleListener.STATUS_DOWNLOAD_FAILURE:
			case ModuleListener.STATUS_NO_UPDATE:
				ModuleFunctions.requestSystemUpdateAbort(status);
				break;
			default:
				println("中断理由が間違っています。");
				Usage();
				break;
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage =	"システムアップデートの中断を要求します。\r\n" +
					"systemupdateabort <status>\r\n" +
					"\tstatus:中断理由\r\n" +
					"\t0:STATUS_NO_UPDATE\r\n" +
					"\t1:STATUS_DOWNLOAD_FAILURE\r\n";
			print(usage);
		}

	}

	/**
	 * システムアップデートのキャンセルの可否を指定します。
	 */
	private class CommandCancelSystemUpdate implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				boolean ena = ModuleProperties.getInstance().getCancelSystemUpdate();
				println(String.valueOf(ena));
				return true;
			}
			ModuleProperties.getInstance().setCancelSystemUpdate(Boolean.valueOf(params.get(0)));
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage =	"システムアップデートのキャンセルの可否を指定します。\r\n" +
					"cancelsystemupdate <bool>\r\n";
			print(usage);
		}

	}

	/**
	 * システムアップデートの実行状態の変更
	 */
	private class CommandUpdateSystemStatus implements TerminalCommand {
		String[] strStatus = new String[5];
		CommandUpdateSystemStatus() {
			strStatus[0] = "IDLE";
			strStatus[1] = "DOWNLOADING";
			strStatus[2] = "DOWNLOAD_COMPLETE";
			strStatus[3] = "REWRITING";
			strStatus[4] = "SENDING_COMPINFO";
		}
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.isEmpty()) {
				int status;
				try {
					status = moduleMan.getSystemUpdateStatus();
				} catch (ModuleException e) {
					return false;
				}
				println(strStatus[status]);
			}
			else {
				String str = params.get(0);
				for (int i = 0; i < strStatus.length; i++) {
					str.toUpperCase();
					if (str.equals(strStatus[i]) || str.equals(String.valueOf(i))) {
						int prev = ModuleProperties.getInstance().getSystemUpdateStatus();
						ModuleProperties.getInstance().setSystemUpdateStatus(i);
						String msg = strStatus[prev] + "→" + strStatus[i] + "\r\n";
						println(msg);
						return true;
					}
				}
				String err = "パラメータ(" + str + ")が間違っています。";
				println(err);
			}
			return true;
		}
		public void Usage() throws IOException{
			String usage =	"システムアップデートの実行状態を設定します。\r\n" +
							"updatestatus <status>\r\n" +
							"\tstatus : 実行状態\r\n" +
							"\t実行状態は、以下の数値または文字列で指定します。\r\n";
			print(usage);
			for (int i = 0; i < strStatus.length; i++) {
				usage = String.valueOf(i) + " : " + strStatus[i];
				println(usage);
			}
		}
	}
	/**
	 * アンテナ状態の変更
	 */
	private class CommandSignalStatus implements TerminalCommand {
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() > 0) {
				int status = -1;
				try {
					status = Integer.valueOf(params.get(0));
				} catch (NumberFormatException e) {
					Usage();
				}
				if ((status >= 0) && (status <= 3)) {
					int prev = ModuleProperties.getInstance().getSignalStatus();
					ModuleProperties.getInstance().setSignalStatus(status);
					String msg = String.valueOf(prev) + "→" + String.valueOf(status);
					println(msg);
				}
				else {
					Usage();
				}
			}
			else {
				Usage();
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"アンテナ状態を変更します。\r\n" +
							"\tsignalstatus <status>\r\n" +
							"\tstatus：0,1,2,3を指定し、0は圏外を表します。\r\n";
			print(usage);
		}
	}
	/**
	 * エリア状態の変更
	 */
	private class CommandAreaInfo implements TerminalCommand {
		String[] area;
		int[] mask;
		public CommandAreaInfo() {
			area = new String[4];
			area[0] = "OUTSIDE";
			area[1] = "HSUPA";
			area[2] = "HSDPA";
			area[3] = "FOMA";
			mask = new int[4];
			mask[0] = 0;
			mask[1] = 1;
			mask[2] = 2;
			mask[3] = 4;
		}
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			int info = -1;
			if (params.size() == 0) {
				Usage();
				return true;
			}
			info = Integer.valueOf(params.get(0));
			if (info < 0 || info > 7) {
				Usage();
				return true;
			}
			int prev = ModuleProperties.getInstance().getAreaInfo();
			ModuleProperties.getInstance().setAreaInfo(info);
			String msg = String.valueOf(prev) + "→" + String.valueOf(info);
			println(msg);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"エリア状態を変更します。\r\n" +
					"\tareainfo <area>\r\n" +
					"\tarea：以下の値のORを指定し、0は圏外を表します。\r\n" ;
			print(usage);
			for (int i = 0; i < area.length; i++) {
				println("\t" + String.valueOf(mask[i]) + "：" + area[i]);
			}
		}

	}
	/**
	 * 規制状態の変更
	 */
	private class CommandRegulation implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			int reg = Integer.valueOf(params.get(0));
			if (reg < 0 || reg > 3) {
				Usage();
				return true;
			}
			int prev = ModuleProperties.getInstance().getRegulation();
			ModuleProperties.getInstance().setRegulation(reg);
			String msg = String.valueOf(prev) + "→" + String.valueOf(reg);
			println(msg);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"規制状態を変更します。\r\n" +
					"\treguration <reg>\r\n" +
					"\treg：以下の値のORを指定します。\r\n" +
					"\t0：NONE\r\n" + "\t1：VOICE\r\n" + "\t2：PACKET\r\n";
			print(usage);
		}

	}
	/**
	 * 受信電力指標の変更
	 */
	private class CommandReceivedPower implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			int power = Integer.valueOf(params.get(0));
			if (power < 0 || power > 75) {
				Usage();
				return true;
			}
			int prev = ModuleProperties.getInstance().getRecievedPower();
			ModuleProperties.getInstance().setRecievedPower(power);
			String msg = String.valueOf(prev) + "→" + String.valueOf(power);
			println(msg);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"受信電力指標を変更します。\r\n" +
					"\treceivedpower <power>\r\n" +
					"\tpower：受信電力指標を0～75の値で指定します。\r\n";
			print(usage);
		}
	}
	/**
	 * UIM挿入状態の変更
	 *
	 */
	private class CommandUIM implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				println("UIM = " + String.valueOf(ModuleProperties.getInstance().getUIM()));
				return true;
			}
			boolean enable = Boolean.valueOf(params.get(0));
			ModuleProperties.getInstance().setUIM(enable);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"UIMの挿入状態を変更します。\r\n" +
					"\tUIM <enable>\r\n" +
					"\tenable：UIMの挿入状態をTRUE/FALSEで指定します。TRUE: 挿入状態\r\n";
			print(usage);
		}

	}
	/**
	 * ModuleExceptionの発生の有無を設定
	 */
	private class CommandModuleException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setModuleException(ena);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ModuleProperties.getInstance().setModuleException(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"ModuleExceptionの発生の有無を指定します。\r\n" +
					"\tmoduleexception <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、ModuleExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * CommunicationFailureExceptionの発生の有無を設定
	 */
	private class CommandCommunicationFailureException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setCommunicationFailureException(ena);
				return true;
			}
			int status = Integer.valueOf(params.get(1));
			if (params.size() == 2) {
				ModuleProperties.getInstance().setCommunicationFailureException(ena, status);
				return true;
			}
			String message = params.get(2);
			for (int i = 3; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ModuleProperties.getInstance().setCommunicationFailureException(ena, status, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"CommunicationFailureExceptionの発生の有無を指定します。\r\n" +
					"\tCommunicationFailureException <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、ModuleExceptionが発生します。\r\n";
			print(usage);
		}

	}
	/**
	 * DeviceExceptionの発生の有無を設定
	 */
	private class CommandDeviceException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setDeviceException(ena);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ModuleProperties.getInstance().setDeviceException(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"DeviceExceptionの発生の有無を指定します。\r\n" +
					"\tDeviceException <bool> <message>\r\n" +
					"\tbool：TRUEが指定された場合、DeviceExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * IOExceptionの発生の有無を設定
	 */
	private class CommandIOException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ConnectionProperties.getInstance().setIOExcepton(ena, null);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ConnectionProperties.getInstance().setIOExcepton(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"IOExceptionの発生の有無を指定します。\r\n" +
					"\tIOException <bool> <message>\r\n" +
					"\tbool：TRUEが指定された場合、DeviceExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * ExternalStatusExceptionの発生の有無を設定
	 */
	private class CommandExternalStatusException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			int status = ExternalStatusException.UNDEFINED;
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setExternalStatusException(ena);
				return true;
			}
			status = Integer.valueOf(params.get(1));
			if (params.size() == 2) {
				ModuleProperties.getInstance().setExternalStatusException(ena, status, null);
				return true;
			}
			String message = params.get(2);
			for (int i = 3; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ModuleProperties.getInstance().setExternalStatusException(ena, status, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"ExternalStatusExceptionの発生の有無を指定します。\r\n" +
					"\tCommunicationExternalStatusException <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、ExternalStatusExceptionが発生します。\r\n";
			print(usage);
		}

	}
	/**
	 * IllegalSettingExceptionの発生の有無を設定
	 */
	private class CommandIllegalSettingException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setIllegalSettingException(ena);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ModuleProperties.getInstance().setIllegalSettingException(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"IllegalSettingExceptionの発生の有無を指定します。\r\n" +
					"\tIllegalSettingException <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、IllegalSettingExceptionが発生します。\r\n";
			print(usage);
		}

	}
	/**
	 * RegulationExceptionの発生の有無を設定
	 */
	private class CommandRegulationException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setRegulationException(ena);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ModuleProperties.getInstance().setRegulationException(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"RegulationExceptionの発生の有無を指定します。\r\n" +
					"\tRegulationException <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、RegulationExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * USSDServiceExceptionの発生の有無を設定
	 */
	private class CommandUSSDServiceException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				NetProperties.getInstance().setUSSDServiceException(ena, null);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			NetProperties.getInstance().setUSSDServiceException(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"USSDServiceExceptionの発生の有無を指定します。\r\n" +
					"\tUSSDServiceException <bool> <message>\r\n" +
					"\tbool：TRUEが指定された場合、USSDServiceExceptionが発生します。\r\n";
			print(usage);
		}
	}
	/**
	 * SecureHandshakeExceptionの発生の有無を設定
	 */
	private class CommandSecureHandshakeException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			int status = SecureHandshakeException.UNDEFINED;
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				SSLProperties.getInstance().setSecureHandshakeException(ena, status, null);
				return true;
			}
			status = Integer.valueOf(params.get(1));
			if (params.size() == 2) {
				SSLProperties.getInstance().setSecureHandshakeException(ena, status, null);
				return true;
			}
			String message = params.get(2);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			SSLProperties.getInstance().setSecureHandshakeException(ena, status, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"SecureHandshakeExceptionの発生の有無を指定します。\r\n" +
					"\tSecureHandshakeException <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、SecureHandshakeExceptionが発生します。\r\n";
			print(usage);
		}

	}
	/**
	 * IllegalCertificateExceptionの発生の有無を設定
	 */
	private class CommandIllegalCertificateException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				SSLProperties.getInstance().setIllegalCertificateException(ena, null);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			SSLProperties.getInstance().setIllegalCertificateException(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"IllegalCertificateExceptionの発生の有無を指定します。\r\n" +
					"\tIllegalCertificateException <bool> <message>\r\n" +
					"\tbool：TRUEが指定された場合、IllegalCertificateExceptionが発生します。\r\n";
			print(usage);
		}
	}

	/**
	 * TimerExceptionの発生の有無を設定
	 */
	private class CommandTimerException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setTimerExcepiton(ena, null);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ModuleProperties.getInstance().setTimerExcepiton(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"TimerExceptionの発生の有無を指定します。\r\n" +
					"\tTimerException <bool> <message>\r\n" +
					"\tbool：TRUEが指定された場合、TimerExceptionが発生します。\r\n";
			print(usage);
		}
	}

	/**
	 * ConnectionExceptionの発生の有無を設定
	 */
	private class CommandConnectionException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				ModuleProperties.getInstance().setTimerExcepiton(ena, null);
				return true;
			}
			String message = params.get(1);
			for (int i = 2; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			ConnectionProperties.getInstance().setConnectionException(ena, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"ConnectionExceptionの発生の有無を指定します。\r\n" +
					"\tConnectionException <bool> <message>\r\n" +
					"\tbool：TRUEが指定された場合、ConnectionExceptionが発生します。\r\n";
			print(usage);
		}
	}

	/**
	 * SessionExceptionの発生の有無を設定
	 */
	private class CommandSessionException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				NetProperties.getInstance().setSessionException(ena, SessionException.UNDEFINED, null);
				return true;
			}
			int status = Integer.valueOf(params.get(1));
			if (params.size() == 2) {
				NetProperties.getInstance().setSessionException(ena, status, null);
				return true;
			}
			String message = params.get(2);
			for (int i = 3; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			NetProperties.getInstance().setSessionException(ena, status, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"SessionExceptionの発生の有無を指定します。\r\n" +
					"\tsessionexception <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、SessionExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * MailExceptionの発生の有無を設定
	 */
	private class CommandMailException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				NetProperties.getInstance().setMailException(ena, MailException.UNDEFINED, null);
				return true;
			}
			int status = Integer.valueOf(params.get(1));
			if (params.size() == 2) {
				NetProperties.getInstance().setMailException(ena, status, null);
				return true;
			}
			String message = params.get(2);
			for (int i = 3; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			NetProperties.getInstance().setMailException(ena, status, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"MailExceptionの発生の有無を指定します。\r\n" +
					"\tmailexception <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、MailExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * ShortMessageExceptionの発生の有無を設定
	 */
	private class CommandShortMessageException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				NetProperties.getInstance().setShortMessageException(ena, ShortMessageException.UNDEFINED, null);
				return true;
			}
			int status = Integer.valueOf(params.get(1));
			if (params.size() == 2) {
				NetProperties.getInstance().setShortMessageException(ena, status, null);
				return true;
			}
			String message = params.get(2);
			for (int i = 3; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			NetProperties.getInstance().setShortMessageException(ena, status, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"ShortMessageExceptionの発生の有無を指定します。\r\n" +
					"\tshortmessageexception <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、SessionExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * LocationExceptionの発生の有無を設定
	 */
	private class CommandLocationException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			if (params.size() == 1) {
				LocationProperties.getInstance().setLocationException(ena, LocationException.UNDEFINED, null);
				return true;
			}
			int status = Integer.valueOf(params.get(1));
			if (params.size() == 2) {
				LocationProperties.getInstance().setLocationException(ena, status, null);
				return true;
			}
			String message = params.get(2);
			for (int i = 3; i < params.size(); i++) {
				message += " ";
				message += params.get(i);
			}
			LocationProperties.getInstance().setLocationException(ena, status, message);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"LocationExceptionの発生の有無を指定します。\r\n" +
					"\tlocationexception <bool> <status> <message>\r\n" +
					"\tbool：TRUEが指定された場合、LocationExceptionが発生します。\r\n";
			print(usage);
		}
	}
	/**
	 * LoadableModuleExceptionの発生の有無を設定
	 */
	private class CommandLoadableModuleException implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			boolean ena = Boolean.valueOf(params.get(0));
			IOProperties.getInstance().setLoadableModuleExcepiton(ena, 0);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"LoadableModuleExceptionの発生の有無を指定します。\r\n" +
					"\tLoadableModuleException <bool>\r\n" +
					"\tbool：TRUEが指定された場合、LoadableModuleExceptionが発生します。\r\n";
			print(usage);
		}

	}

	/**
	 * 着信呼の接続を行います。
	 */
	private class CommandReceive implements TerminalCommand {
		public CommandReceive() {
		}
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			else {
				String key = params.get(0);
				if (!sessionMap.containsKey(key)) {
					println("セッション(" + key + ")は存在しません");
					return true;
				}
				Session session = sessionMap.get(key);
				String dest = session.getDestination();
				if (NetProperties.isTelephonNumber(dest)) {
					VoiceControllerImpl voiceController;
					try {
						voiceController = (VoiceControllerImpl) ModuleManager.getInstance().getVoiceController();
						voiceController.receive((VoiceSession) session);
					} catch (ModuleException e) {
						// TODO 自動生成された catch ブロック
						e.printStackTrace();
					}
				}
				else {
					PacketControllerImpl packetController;
					try {
						packetController = (PacketControllerImpl) ModuleManager.getInstance().getPacketController();
						packetController.receive(session);
					} catch (SessionException e) {
						e.printStackTrace();
						return false;
					} catch (CommunicationFailureException e) {
						e.printStackTrace();
						return false;
					} catch (IllegalSettingException e) {
						e.printStackTrace();
						return false;
					} catch (ModuleException e) {
						e.printStackTrace();
						return false;
					}

				}
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"着信を行います\r\n" +
					"\treceive <session>\r\n" +
					"\tsession: セッションを指定します。\n";
			print(usage);

		}
	}
	/**
	 * 着信呼の接続を行います。
	 */
	private class CommandSend implements TerminalCommand {
		public CommandSend() {
		}
		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			else {
				Session session;
				String key = params.get(0);
				if (NetProperties.isTelephonNumber(key)) {
					println("音声セッション：" + key);
					VoiceControllerImpl voiceController;
					try {
						voiceController = (VoiceControllerImpl) ModuleManager.getInstance().getVoiceController();
						if (sessionMap.containsKey(key)) {
							session = sessionMap.get(key);
						}
						else {
							session = voiceController.createSession(key);
							sessionMap.put(key, session);
						}
						if (!voiceController.incomingCall(session)) {
							println("着信できません。");
							sessionMap.remove(key);
						}
					} catch (SessionException e1) {
						println("着信を拒否されました。");
					} catch (IllegalStateException e2) {
						println("IllegalStateException");
						e2.printStackTrace();
					} catch (ModuleException e) {
						// TODO 自動生成された catch ブロック
						e.printStackTrace();
					}
				}
				else {
					// パケットセッション
					println("パケットセッション：" + key);
					PacketControllerImpl packetController;
					try {
						packetController = (PacketControllerImpl) ModuleManager.getInstance().getPacketController();
					} catch (ModuleException e) {
						return false;
					}
					try {
						session = packetController.createSession(key);
						sessionMap.put(key, session);
						if (!packetController.incomingCall(session)) {
							println("着信できません。");
							sessionMap.remove(key);
						}
					} catch (SessionException e1) {
						println("着信を拒否されました。");
					}
				}
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"発信を行います\r\n" +
					"\tsend <session>\r\n" +
					"\tsession: セッションを指定します。\r\n";
			print(usage);

		}
	}
	/**
	 * 呼の切断を行います。
	 */
	private class CommandDisconnect implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			else {
				String key = params.get(0);
				if (!sessionMap.containsKey(key)) {
					println("セッション(" + key + ")は存在しません");
					return true;
				}
				Session session = sessionMap.get(key);
				if (NetProperties.isTelephonNumber(key)) {
					VoiceControllerImpl voiceController;
					try {
						voiceController = (VoiceControllerImpl) ModuleManager.getInstance().getVoiceController();
						voiceController.disconnect((VoiceSession) session);
					} catch (ModuleException e) {
						// TODO 自動生成された catch ブロック
						e.printStackTrace();
					} catch (SessionException e) {
						// TODO 自動生成された catch ブロック
						e.printStackTrace();
					}
				} else {
					PacketControllerImpl packetController;
					try {
						packetController = (PacketControllerImpl) ModuleManager.getInstance().getPacketController();
						packetController.disconnect(session);
					} catch (ModuleException e) {
						return false;
					} catch (SessionException e) {
						return false;
					} catch (CommunicationFailureException e) {
						return false;
					} finally {
						sessionMap.remove(key);
					}
				}
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"切断を行います\r\n" +
					"\tdisconnect <session>\r\n" +
					"\tsession: セッションを指定します。\r\n";
			print(usage);
		}
	}
	/**
	 * USSDの送信を行います。
	 */
	private class CommandUSSDSend implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			String message = params.get(0);
			if (!USSDFunctions.received(message)) {

			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"USSDの送信を行います\r\n" +
					"\tussdsend <message>\r\n" +
					"\tmessage: メッセージ\r\n";
			print(usage);

		}

	}

	/**
	 * 音声ガイダンスの送信を行います。
	 */
	private class CommandGuidance implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			else {
				String key = params.get(0);
				if (!sessionMap.containsKey(key)) {
					println("セッション(" + key + ")は存在しません");
					return true;
				}
				VoiceSession session = (VoiceSession)sessionMap.get(key);
				voiceController.guidance(session);
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"音声ガイダンスの送信を行います\r\n" +
					"\tguidance <session>\r\n" +
					"\tguidance: セッションを指定します。\r\n";
			print(usage);
		}
	}
	/**
	 * SMSの送信を行います。
	 *
	 */
	private class CommandSMSSend implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			if (params.size() == 0) {
				Usage();
				return true;
			}
			try {
				ShortMessageFunctions.receiveSMS(params.get(0));
			} catch (DeviceException e) {
				println("DeviceException");
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"SMSの送信を行います\r\n" +
					"\tsmssend <filename>\r\n" +
					"\tfilename: ファイル名\r\n";
			print(usage);
		}

	}

	/**
	 * I2Cの送信を行います。
	 *
	 */
	private class CommandI2CSend implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			I2CConnection i2c;
			try {
				i2c = (I2CConnection)ModuleManager.getInstance().getConnectionManager().getConnection(ConnectionManager.CONNECTION_TYPE_I2C);
			} catch (IllegalArgumentException e1) {
				return false;
			} catch (ModuleException e1) {
				return false;
			}
			List<I2CSpec> specList = i2c.getI2CSpec();
			I2CSlave slave = null;
			try {
				slave = i2c.getI2CSlave(specList.get(0), 0);
			} catch (IllegalStateException e) {
				println("I2Cデバイスはマスタとして登録されています。");
				return true;
			} catch (ConnectionException e) {
				return true;
			}
//			slave.receiveData();
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"I2Cクライアントへの送信を行います\r\n" +
					"\tI2CSend\r\n";
			print(usage);
		}

	}

	/**
	 * AD値の設定を行います。
	 *
	 */
	private class CommandAD implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			ADConverter ad;
			try {
				ad = (ADConverter)ModuleManager.getInstance().getConnectionManager().getConnection(ConnectionManager.CONNECTION_TYPE_ADCONVERTER);
			} catch (IllegalArgumentException e) {
				return false;
			} catch (ModuleException e) {
				return false;
			}
			List<ADChannel> chList = ad.getADChannelList();
			if (params.size() == 0) {
				for (int i = 0; i < chList.size(); i++) {
					ADChannel ch = chList.get(i);
					println(String.valueOf(i) + ":" + ch.getChannelName() + ":" + String.valueOf(ch.getResolution()));
				}
				return true;
			}
			int no = Integer.valueOf(params.get(0));
			if (params.size() == 1) {
				try {
					ADChannel ch = chList.get(no);
					println(String.valueOf(no) + ":" + ch.getChannelName() + ":" + String.valueOf(ch.getResolution() + ":" + String.valueOf(ad.getAD(ch))));
				} catch (ConnectionException e) {

				}
				return true;
			}
			int value = Integer.valueOf(params.get(1));
			ADConverterFunctions.setAD(no, value);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"AD値のセットを行います。\r\n" +
					"\tAD <ch> <value>\r\n" +
					"\t<ch>：チャネル番号\r\n" +
					"\t<value>：AD値\r\n";
			print(usage);
		}

	}
	/**
	 * GPIOのポート値の設定を行います。
	 *
	 */
	private class CommandGPIO implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			GPIOConnection gpio;
			try {
				gpio = (GPIOConnection)ModuleManager.getInstance().getConnectionManager().getConnection(ConnectionManager.CONNECTION_TYPE_GPIO);
			} catch (IllegalArgumentException e) {
				return false;
			} catch (ModuleException e) {
				return false;
			}
			List<Integer> inport = gpio.getAvailableInPort();
			List<Integer> outport = gpio.getAvailableOutPort();
			if (params.size() == 0) {
				if (inport.size() > 0) {
					print("GPIO Inport : " + String.valueOf(inport.get(0)));
					for (int i = 1; i < inport.size(); i++) {
						print(", " + String.valueOf(inport.get(i)));
					}
				}
				if (outport.size() > 0) {
					print("\r\nGPIO Outport : " + String.valueOf(outport.get(0)));
					for (int i = 1; i < outport.size(); i++) {
						print(", " + String.valueOf(outport.get(i)));
					}
					println("");
				}
				return true;
			}
			int no = Integer.valueOf(params.get(0));
			boolean value;
			if (params.size() == 1) {
				try {
					if (inport.contains(no)) {
						value = gpio.receiveControlSignal(no);
						println("GPIO Inport[" + String.valueOf(no) + "] = " + String.valueOf(value));
					}
					else if (outport.contains(no)) {
						value = gpio.getControlSignal(no);
						println("GPIO Outport[" + String.valueOf(no) + "] = " + String.valueOf(value));
					}
				}
				catch (ConnectionException e) {

				}
				return true;
			}
			value = Boolean.valueOf(params.get(1));
			if (inport.contains(no)) {
				GPIOFunctions.setReceiveControlSignal(no, value);
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"GPIOのポート値のセットを行います。\r\n" +
					"\tGPIO <port> <signal>\r\n" +
					"\t<port>：ポート番号\r\n" +
					"\t<signal>：TRUE/FALSE\r\n";
			print(usage);
		}

	}
	/**
	 * SPIの通信を開始します。
	 *
	 */
	private class CommandSPIStart implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			SPIFunctions.start();
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"SPI通信を開始します。\r\n" +
							"\tSPIStart\r\n";
			print(usage);
		}

	}
	/**
	 * SPIの通信を停止します。
	 *
	 */
	private class CommandSPIStop implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			SPIFunctions.stop();
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"SPI通信を停止します。\r\n" +
					"\tSPIStart\r\n";
			print(usage);

		}

	}
	/**
	 * 現在位置の表示／設定を行います。
	 *
	 */
	private class CommandLocation implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			Location location;
			Degree latitude;
			Degree longitude;
			if (params.size() != 6) {
				location = LocationProperties.getInstance().getLocation();
				latitude = location.getLatitude();
				longitude = location.getLongitude();
				println("緯度: " + String.valueOf(latitude.getDegreePart()) + ", " + String.valueOf(latitude.getMinutePart()) + ", " + String.valueOf(latitude.getCentisecondPart()));
				println("経度: " + String.valueOf(longitude.getDegreePart()) + ", " + String.valueOf(longitude.getMinutePart()) + ", " + String.valueOf(longitude.getCentisecondPart()));
				return true;
			}
			latitude = new Degree(Integer.valueOf(params.get(0)), Integer.valueOf(params.get(1)), Integer.valueOf(params.get(2)));
			longitude = new Degree(Integer.valueOf(params.get(3)), Integer.valueOf(params.get(4)), Integer.valueOf(params.get(5)));
			location = new Location(latitude, longitude);
			LocationProperties.getInstance().setLocation(location);
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"現在位置の表示／設定を行います。\r\n" +
							"\tlocation <latitude> <longitude>\r\n" +
							"\t<latitude>:  緯度を指定します。\r\n" +
							"\t<longitude>: 経度を指定します。\r\n";
			print(usage);
		}
	}

	/**
	 * 位置提供を行います。
	 *
	 */
	private class CommandProvideLocation implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			int status = LocationProvider.STATUS_EMERGENCY;
//			if (params.size() == 0) {
//				Usage();
//				return true;
//			}
//			if (params.size() > 1) {
//				status = Integer.valueOf(params.get(1));
//			}
			try {
				LocationProviderImpl provider = (LocationProviderImpl) ModuleManager.getInstance().getLocationProvider(LocationProvider.METHOD_AUTO);
//				if (provider.ProvideLocation(params.get(0), status) == null) {
				if (provider.ProvideLocation(status) == null) {
					println("位置提供に失敗しました。");
				}
			} catch (ModuleException e) {
				e.printStackTrace();
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"位置提供を行います。\r\n" +
							"\tprovidelocation\r\n";
			print(usage);
		}
	}

	/**
	 * 位置提供を行います。
	 *
	 */
	private class CommandLocationStatus implements TerminalCommand {

		@Override
		public boolean CommandProc(List<String> params) throws IOException {
			int status = 0;
			try {
				status = ModuleManager.getInstance().getLocationProvider(LocationProvider.METHOD_AUTO).getStatus();
			} catch (ModuleException e) {
				// TODO 自動生成された catch ブロック
				e.printStackTrace();
			} catch (LocationException e) {
				// TODO 自動生成された catch ブロック
				e.printStackTrace();
			}
			if (params.size() == 0) {
				println("status = " + String.valueOf(status));
				return true;
			}
			status = Integer.valueOf(params.get(0));
			try {
				LocationProviderImpl provider = (LocationProviderImpl) ModuleManager.getInstance().getLocationProvider(LocationProvider.METHOD_AUTO);
				switch (status) {
				case LocationProvider.STATUS_IDLE:
				case LocationProvider.STATUS_MEASUREMENT:
				case LocationProvider.STATUS_EMERGENCY:
					provider.setStatus(status);
					break;
				default:
					println("測位状態が間違っています。");
					Usage();
					break;
				}
			} catch (ModuleException e) {
				// TODO 自動生成された catch ブロック
				e.printStackTrace();
			}
			return true;
		}

		@Override
		public void Usage() throws IOException {
			String usage = 	"GPSの測位状態を表示／設定します。\r\n" +
							"\tlocationstatus <status>\r\n" +
							"\t0:STATUS_IDLE\r\n"+
							"\t1:STATUS_MESURMENT\r\n"+
//							"\t2:STATUS_NOTIFY\r\n"+
//							"\t3:STATUS_PROVIDE\r\n"+
							"\t4:STATUS_EMERGENCY\r\n";
			print(usage);
		}
	}

	/**
	 * ターミナルI/Fコマンド・インターフェイス
	 */
	private interface TerminalCommand {
		public abstract boolean CommandProc(List<String> params)  throws IOException;
		public abstract void Usage() throws IOException;
	}

	public class LogMessageListener implements LogListener {
		@Override
		public void onMessage(String message) {
			println(message);
		}

	}
	private class TerminalUSSDListener implements USSDServiceTerminalListener {

		@Override
		public void onSend(String message) {
			println("USSD：" + message);
		}

	}
	/**
	 * セッションリスナクラス
	 *
	 */
	private class TerminalSessionListener implements ModemControllerTerminalListener {
		@Override
		public void onSend(ModemController controller, Session session) {
			if (controller.equals(voiceController)) {
				VoiceSession voice = (VoiceSession)session;
				String destination = voice.getDestination();
				println("音声発信：" + destination);
				sessionMap.put(destination, voice);
			}
			else if (controller.equals(packetController)) {
				PacketSession packet = (PacketSession)session;
				String destination = packet.getDestination();
				println("パケット発信：" + destination);
				if (packet.getPDPTypeInfo().getPDPType() == PDPTypeInfo.PDP_TYPE_IP) {
					IPSettingInfo setting = (IPSettingInfo)packet.getPDPTypeInfo();
					print("IP:  autthentication = ");
					switch (setting.getIPAuthentication()) {
					case IPSettingInfo.IP_AUTHENTICATION_NONE:
						println("NONE");
						break;
					case IPSettingInfo.IP_AUTHENTICATION_CHAP:
						println("CHAP");
						println("    username = " + setting.getUserName());
						println("    password = " + setting.getPassword());
						break;
					case IPSettingInfo.IP_AUTHENTICATION_PAP:
						println("PAP");
						println("    username = " + setting.getUserName());
						println("    password = " + setting.getPassword());
						break;
					}
					print("    Allocating IP = ");
					switch (setting.getAllocatingIP()) {
					case IPSettingInfo.ALLOCATING_IP_STATIC:
						println("STATIC");
						println("    IP address = " + setting.getIPAddress());
						break;
					case IPSettingInfo.ALLOCATING_IP_DYNAMIC:
						println("DYNAMIC");
						break;
					}
					print("    Allocating DNS = ");
					switch (setting.getAllocatingDNS()) {
					case IPSettingInfo.ALLOCATING_IP_STATIC:
						println("STATIC");
						println("    primary DNS   = " + setting.getPrimaryDNS());
						println("    secondary DNS = " + setting.getSecondaryDNS());
						break;
					case IPSettingInfo.ALLOCATING_IP_DYNAMIC:
						println("DYNAMIC");
						break;
					}
				}
				else if (packet.getPDPTypeInfo().getPDPType() == PDPTypeInfo.PDP_TYPE_PPP) {
					PPPSettingInfo setting = (PPPSettingInfo)packet.getPDPTypeInfo();
					print("PPP: autthentication = ");
					switch (setting.getPPPAuthentication()) {
					case PPPSettingInfo.PPP_AUTHENTICATION_NONE:
						println("NONE");
						break;
					case PPPSettingInfo.PPP_AUTHENTICATION_CHAP:
						println("CHAP");
						println("    username = " + setting.getUserName());
						println("    password = " + setting.getPassword());
						break;
					case PPPSettingInfo.PPP_AUTHENTICATION_PAP:
						println("PAP");
						println("    username = " + setting.getUserName());
						println("    password = " + setting.getPassword());
						break;
					}
					print("    Allocating IP = ");
					switch (setting.getAllocatingIP()) {
					case PPPSettingInfo.ALLOCATING_IP_STATIC:
						println("STATIC");
						println("    IP address = " + setting.getIPAddress());
						break;
					case PPPSettingInfo.ALLOCATING_IP_DYNAMIC:
						println("DYNAMIC");
						break;
					}
					print("    Allocating DNS = ");
					switch (setting.getAllocatingDNS()) {
					case PPPSettingInfo.ALLOCATING_IP_STATIC:
						println("STATIC");
						println("    primary DNS   = " + setting.getPrimaryDNS());
						println("    secondary DNS = " + setting.getSecondaryDNS());
						break;
					case PPPSettingInfo.ALLOCATING_IP_DYNAMIC:
						println("DYNAMIC");
						break;
					}
				}
				sessionMap.put(destination, packet);
			}
		}
		/**
		 * 切断
		 */
		@Override
		public void onDisconnect(ModemController controller, Session session) {
			if (controller.equals(voiceController)) {
				VoiceSession voice = (VoiceSession)session;
				String destination = voice.getDestination();
				println("音声切断：" + destination);
				sessionMap.remove(destination);
			}
			else if (controller.equals(packetController)) {
				PacketSession packet = (PacketSession)session;
				String destination = packet.getDestination();
				println("パケット切断：" + destination);
				sessionMap.remove(destination);
			}
		}
		/**
		 * 保留
		 */
		@Override
		public void onHold(ModemController controller, Session session) {
			if (controller.equals(voiceController)) {
				VoiceSession voice = (VoiceSession)session;
				String destination = voice.getDestination();
				println("保留：" + destination);
			}
			else if (controller.equals(packetController)) {
				// nop
			}
		}
		/**
		 * 着信応答
		 */
		@Override
		public void onReceive(ModemController controller, Session session) {
			if (controller.equals(voiceController)) {
				VoiceSession voice = (VoiceSession)session;
				String destination = voice.getDestination();
				println("音声応答：" + destination);
			}
			else if (controller.equals(packetController)) {
				PacketSession packet = (PacketSession)session;
				String destination = packet.getDestination();
				println("パケット応答：" + destination);
			}
		}
		/**
		 * DTMF送信
		 */
		@Override
		public void onDTMF(ModemController controller, Session session,
				String message) {
			if (controller.equals(voiceController)) {
				VoiceSession voice = (VoiceSession)session;
				String destination = voice.getDestination();
				println("DTMF：" + destination + "：" + message);
			}
			else if (controller.equals(packetController)) {
				// nop
			}
		}
		/**
		 * 接続拒否
		 */
		@Override
		public void onReject(ModemController controller, Session session) {
			if (controller.equals(voiceController)) {
				VoiceSession voice = (VoiceSession)session;
				String destination = voice.getDestination();
				println("接続拒否：" + destination);
			}
			else if (controller.equals(packetController)) {

			}
		}
		/**
		 * 留守番電話サービス
		 */
		@Override
		public void onSendAnswering(ModemController controller, Session session) {
			if (controller.equals(voiceController)) {
				VoiceSession voice = (VoiceSession)session;
				String destination = voice.getDestination();
				println("留守番電話：" + destination);
			}
		}
	}
}
