import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import com.docomo_um.io.FFSAccessManager;
import com.docomo_um.io.FFSFile;
import com.docomo_um.io.IOProperties;

public class FFSAccessManagerTest {
	String[] filenames;

	public FFSAccessManagerTest() {
		filenames = new String[10];
		filenames[0] = "ffsfile00.dat";
		filenames[1] = "ffsfile01.dat";
		filenames[2] = "ffsfile02.dat";
		filenames[3] = "ffsfile03.dat";
		filenames[4] = "ffsfile04.dat";
		filenames[5] = "ffsfile05.dat";
		filenames[6] = "ffsfile06.dat";
		filenames[7] = "ffsfile07.dat";
		filenames[8] = "ffsfile08.dat";
		filenames[9] = "ffsfile09.dat";
	}
	/**
	 * 初期化
	 * 既存のFFSファイルおよびディレクトリを削除する。
	 */
	public void Test01_initialize() {
		File file;
		// 既存のFFSファイルの削除
		for (int i = 0; i < filenames.length; i++) {
			file = new File(IOProperties.getInstance().getFFSDir() + filenames[i]);
			if (!file.exists()) {
				continue;
			}
			file.delete();
		}
		// 既存のディレクトリの削除
		file = new File(IOProperties.getInstance().getFFSDir());
		if (file.exists()) {
			file.delete();
		}
	}
	/**
	 * インスタンスの生成を行い、ディレクトリおよび空のファイルが作成されていることを確認
	 *
	 * @throws Exception
	 */
	public void Test01_01() throws Exception {
		FFSAccessManager instance = FFSAccessManager.getInstance();
		System.out.println("TEST01_01: FFS remain size = " + String.valueOf(instance.getRemainSize()));
		for (int i = 0; i < filenames.length; i++) {
			File file = new File(IOProperties.getInstance().getFFSDir() + filenames[i]);
			System.out.println("TEST01_01: check:" + file.getName());
			if (!file.exists()) {
				throw new Exception("TEST01_01: fail");
			}
			System.out.println("TEST01_01: pass:" + file.getName() + " exists");
		}
		System.out.println("TEST01_01: pass");
	}
	/**
	 * ファイルリストを正しく取得できることを確認
	 *
	 * @throws Exception
	 */
	public void Test01_02() throws Exception {
		FFSAccessManager instance = FFSAccessManager.getInstance();
		List<FFSFile> list = instance.getFileList();
		System.out.println("TEST01_02: FileList size = " + String.valueOf(list.size()));
		if (list.size() != 10) {
			throw new Exception("Test01_02: fail");
		}
		for (int i = 0; i < list.size(); i++) {
			FFSFile file = list.get(i);
			String name = file.getName();
			long sz = file.length();
			System.out.println("TEST01_02: name = " + name + ", size = " + String.valueOf(sz));
			if (!name.equals(filenames[i])) {
				throw new Exception("TEST01_02: fail");
			}
			if (file.length() != 0) {
				throw new Exception("TEST01_02: fail");
			}
		}
		System.out.println("TEST01_02: pass");
	}
	/**
	 * 出力ストリームを取得し、書き込みを行えることを確認
	 * @throws Exception
	 */
	public void Test01_03() throws Exception {
		FFSAccessManager instance = FFSAccessManager.getInstance();
		List<FFSFile> list = instance.getFileList();
		FFSFile file = list.get(0);
		OutputStream fos = instance.openOutputStream(file, false);
		byte[] buf = new byte[256];
		for (int i = 0; i < 256; i++) {
			buf[i] = (byte)i;
		}
		int sz1,sz2;
		sz1 = instance.getRemainSize();
		System.out.println("TEST01_03: FFS remain size = " + String.valueOf(sz1));
		System.out.println("TEST01_03: 256 byte write");
		fos.write(buf);
		sz2 = instance.getRemainSize();
		System.out.println("TEST01_03: FFS remain size = " + String.valueOf(sz2));
		if ((sz1 - 256) != sz2) {
			throw new Exception("Test01_03: fail");
		}
		fos.flush();
		fos.close();

		System.out.println("TEST01_03: verify");
		File f = new File(IOProperties.getInstance().getFFSDir() + filenames[0]);
		FileInputStream fis = new FileInputStream(f);
		for (int i = 0; i < 256; i++) {
			int b = fis.read();
			System.out.print(String.valueOf(b) + " ");
			if (b == -1) {
				throw new Exception("TEST01_03: fail");
			}
			if (buf[i] != (byte)b) {
				throw new Exception("TEST01_03: fail");
			}
		}
		fis.close();
		System.out.println("\nTEST01_03: pass");
	}
	/**
	 * 入力ストリーム取得し、ファイルを読み込めることを確認
	 *
	 * @throws Exception
	 */
	public void Test01_04() throws Exception {
		FFSAccessManager instance = FFSAccessManager.getInstance();
		List<FFSFile> list = instance.getFileList();
		FFSFile file = list.get(0);
		InputStream fis = instance.openInputStream(file);
		byte[] buf = new byte[256];
		int sz = fis.read(buf);
		System.out.println("TEST01_04: read size = " + String.valueOf(sz));
		if (sz == -1) {
			throw new Exception("TEST01_04: fail");
		}
		if (sz != 256) {
			throw new Exception("TEST01_04: fail");
		}
		System.out.println("TEST01_04: verify");
		for (int i = 0; i < 256; i++) {
			System.out.print(String.valueOf(buf[i]) + " ");
			if (buf[i] != (byte)i) {
				throw new Exception("TEST01_04: fail");
			}
		}
		fis.close();
		System.out.println("\nTEST01_04: pass");
	}
	/**
	 * 空き容量が0になった場合の書き込み時に例外が発生することを確認
	 *
	 * @throws Exception
	 */
	public void Test01_05() throws Exception {
		FFSAccessManager instance = FFSAccessManager.getInstance();
		List<FFSFile> list = instance.getFileList();
		FFSFile file = list.get(1);
		byte[] buf = new byte[1024];
		OutputStream fos = instance.openOutputStream(file, false);
		int sz = instance.getRemainSize();
		System.out.println("TEST01_05: remain size = " + String.valueOf(sz));
		for (int i = 0; i < 1024; i++) {
			for (int j = 0; j < 1024; j++) {
				buf[i] = (byte)(i % 256);
			}
			try {
				fos.write(buf);
			}
			catch (IOException e) {
				if (i != 1023) {
					throw new Exception("TEST01_05: fail");
				}
			}
			sz = instance.getRemainSize();
			System.out.println("TEST01_05: remain size = " + String.valueOf(sz));
		}
		fos.close();
		System.out.println("TEST01_05: pass");
	}
	/**
	 * 出力ストリーム取得時にappendフラグがfalseの場合、ファイルサイズを0に初期化し、先頭からファイルを書き込めることを確認
	 *
	 * @throws Exception
	 */
	public void Test01_06() throws Exception {
		FFSAccessManager instance = FFSAccessManager.getInstance();
		List<FFSFile> list = instance.getFileList();
		FFSFile file = list.get(1);
		byte[] buf = new byte[1024];
		long sz;
		System.out.println("TEST01_06: remain size = " + String.valueOf(instance.getRemainSize()));
		sz = file.length();
		System.out.println("TEST01_06: file size = " + String.valueOf(sz));
		OutputStream fos = instance.openOutputStream(file, false);
		System.out.println("TEST01_06: remain size = " + String.valueOf(instance.getRemainSize()));
		sz = file.length();
		System.out.println("TEST01_06: file size = " + String.valueOf(sz));
		if (instance.getRemainSize() != 1024 * 1024 - 256) {
			throw new Exception("TEST01_06: fail");
		}
		for (int i = 0; i < 1024; i++) {
			buf[i] = (byte)(i%256);
		}
		fos.write(buf);
		System.out.println("TEST01_06: remain size = " + String.valueOf(instance.getRemainSize()));
		sz = file.length();
		System.out.println("TEST01_06: file size = " + String.valueOf(sz));
		if (instance.getRemainSize() != 1024 * 1024 - 256 - 1024) {
			throw new Exception("TEST01_06: fail");
		}
		fos.close();
		System.out.println("TEST01_06: pass");
	}
	/**
	 * 出力ストリーム取得時にappendフラグがtrueの場合、既存のファイルの末尾から書き込みが行われることを確認
	 *
	 * @throws Exception
	 */
	public void Test01_07() throws Exception {
		FFSAccessManager instance = FFSAccessManager.getInstance();
		List<FFSFile> list = instance.getFileList();
		FFSFile file = list.get(1);
		byte[] buf = new byte[1024];
		int sz = instance.getRemainSize();
		System.out.println("TEST01_07: remain size = " + String.valueOf(sz));
		System.out.println("TEST01_07: file size = " + file.length());
		OutputStream fos = instance.openOutputStream(file, true);
		sz = instance.getRemainSize();
		System.out.println("TEST01_07: remain size = " + String.valueOf(sz));
//		System.out.println("TEST01_07: file size = " + file.length());
		if (sz != 1024 * 1024 - 256 - 1024) {
			throw new Exception("TEST01_07: fail");
		}
 		System.out.println("TEST01_07: file size = " + String.valueOf(file.length()));
		if (file.length() != 1024) {
			throw new Exception("TEST01_07: fail");
		}
		for (int i = 0; i < 1024; i++) {
			buf[i] = (byte)(i%256);
		}
		fos.write(buf);
		fos.flush();
		fos.close();
		sz = instance.getRemainSize();
		System.out.println("TEST01_07: remain size = " + String.valueOf(sz));
		System.out.println("TEST01_07: file size = " + file.length());
		if (sz != 1024 * 1024 - 256 - 2048) {
			throw new Exception("TEST01_07: fail");
		}
		System.out.println("TEST01_07: file size = " + String.valueOf(file.length()));
		if (file.length() != 2048) {
			throw new Exception("TEST01_07: fail");
		}
		System.out.println("TEST01_07: pass");
	}
}
