import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import com.docomo_um.module.ModuleException;
import com.docomo_um.module.ModuleManager;
import com.docomo_um.module.connection.ConnectionException;
import com.docomo_um.module.connection.ConnectionManager;
import com.docomo_um.module.connection.Serial;
import com.docomo_um.module.connection.SerialConnection;
import com.docomo_um.module.connection.SerialListener;
import com.docomo_um.module.connection.SerialSpec;

/**
 * シリアル通信のサンプルクラスです
 * <p>
 * シリアル通信ポートを作成し{@link #setup()}、 通信スピードとリスナを設定する
 * データ送信は、{@link #onReceived(Serial)}でデータを送信する
 * データ受信は、{@link #onReceived(Serial)}で受ける情報を受信する
 * </p>
 * @see SerialListener
 * @see SerialConnection
 * @see SerialSpec
 * @see Serial
 * @see ModuleManager
 * @see ConnectionManager
 */
public class Main implements SerialListener {
	/**
     * プログラムのエントリです
     * @param  なし
	 * @throws なし
     */
	public static void entry() {
		try {
			Thread.sleep(5000);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		// メッセージを出力する
		System.out.println("entry start.");
		// インスタンスを作成し、通信デバイスの情報設定処理をコールする
		new Main().setup();
	}
	/**
	 * 通信デバイスの情報設定処理
	 * <P>
	 * シリアルポートを初期化処理{@link #getSerialConnection()}
	 * シリアルデバイスのインスタンスを生成する{@link #getSerial01(SerialConnection)}
	 * デバイスの通信速度{@link Serial#setCommSpeed(long)}
	 * リスナを登録する{@link Serial#setSerialListener(SerialListener)}
	 * </p>
	 * @param  なし
	 * @throws なし
	 */
	private void setup() {
		System.out.println("setup start.");
		try {
			// SerialConnectionのインスタンスを作成する
			SerialConnection conn = getSerialConnection();
			System.out.println("getSerialConnection");
			// Serialのインスタンスを作成する
			Serial serial = getSerial01(conn);
			System.out.println("getSerial01");
			// シリアルデバイス有効無効の状態を判定する
			if (serial.isEnabled()) {
				// シリアルデバイスが有効の場合
				// シリアルデバイス通信スピードを設定する
				serial.setCommSpeed(19200);
				System.out.println("setCommSpeed");
				// 受信リスナ設定
				serial.setSerialListener(this);
				System.out.println("setSerialListener");
			} else {
				// シリアルデバイスが無効の場合、メッセージを出力
				System.out.println("SerialDevice invalid");
				return;
			}
		} catch (Throwable e) {
			e.printStackTrace();
		} finally {
			System.out.println("process end");
		}
	}
	/**
	 * 有線通信IF取得処理
	 * <p>
	 * ModuleManagerを生成する{@link ModuleManager#getInstance()}
	 * 有線通信IF接続管理クラスのインスタンス作成{@link ModuleManager#getConnectionManager()}
	 * 有線通信IFを取得する{@link ConnectionManager#getConnection(int)}
	 * </p>
	 * @param  なし
	 * @throws ModuleException
	 * @throws IllegalArgumentException
	 */
	private SerialConnection getSerialConnection() throws ModuleException,IllegalArgumentException {
		// 有線通信IF種別をシリアルに設定し、SerialConnectionを取得する
		return (SerialConnection) ModuleManager.getInstance().getConnectionManager().getConnection(ConnectionManager.CONNECTION_TYPE_SERIAL);
	}

	/**
	 * シリアルデバイス有効に設定
	 * <p>
	 * 利用可能なシリアル情報のリスト取得{@link SerialConnection#getSerialSpec()}
	 * シリアルデバイスのインスタンス作成{@link SerialConnection#getSerial(SerialSpec)}
	 * シリアルデバイスの有効に設定{@link Serial#setEnable(boolean)}
	 * </p>
	 * @param  conn　有線通信IFのインスタンス
	 * @throws ConnectionException
	 * @throws NullPointerException
	 */
	private Serial getSerial01(SerialConnection conn) throws ConnectionException,NullPointerException {
		// シリアルデバイスリストを取得する
		List<SerialSpec> specList = conn.getSerialSpec();
		Serial  serial = null;
		int i = 0;
		for (SerialSpec list : specList){
			// USBの場合、以下行を有効化する。UARTの場合はコメントにする
			if (list.getDeviceName().substring(0,3).equals("USB")) {
			// UARTの場合、以下行を有効化する。USBの場合はコメントにする
//			if (list.getDeviceName().substring(0,4).equals("UART")) {
				// SerialConnectionより見つけたシリアルデバイスのインスタンスを取得する
				serial = conn.getSerial(specList.get(i));
				// デバイス名を出力する
				System.out.println("Serial Name :" + list.getDeviceName());
				break;
			}
			++i;
		}
		// シリアルデバイスを有効にする
		serial.setEnable(true);
		// シリアルデバイスのインスタンスを返却する
		return serial;
	}

	/**
	 * シリアルデバイスの送受信処理
	 * <p>
	 * シリアル情報の受信処理
	 * シリアル情報の送信処理
	 * </p>
	 * @param  serial　シリアルデバイスのインスタンス
	 * @throws なし
	 */
	@Override
	public void onReceived(Serial serial) {
		// 受信処理
		// 受信データを格納するために、ByteArrayOutputStreamのインスタンスを作成する
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		// 受信データを取得するために、InputStreamのインスタンスを作成する
		InputStream is = null;
		try {
			System.out.println("recive  start.");
			// 入力ストリームを取得する
			is = serial.getInputStream();
			// 読込みデータのバイト数を定義する
			byte[] b = new byte[1024];
			// 読込みデータレングスを格納する変数を定義する
			int len = 0;
			// 受信データを読み出す
			while((len = is.read(b)) > 0) {
				// 読み出したデータをByteArrayOutputStreamに格納する
				baos.write(b, 0, len);
			}
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (is != null) {
				try {
					// データ受信終了後、InputStreamをクローズする
					is.close();
					System.out.println("InputStream close");
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		// 送信処理
		// データを送信するために、OutputStreamのインスタンスを作成する
		OutputStream os = null;
		try {
			// 出力ストリームを取得する
			os = serial.getOutputStream();
			// データを送信する
			os.write(baos.toByteArray());
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (os != null) {
				try {
					// OutputStreamをクローズする
					os.close();
					System.out.println("OutputStream close");
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * ER信号の状態が変化した場合にコールされるメソッド
	 * <p>
	 * 引数erがfalseの場合、シリアルデバイスを無効に設定
	 * </p>
	 * @param  serial　シリアルデバイスのインスタンス
	 * @param  er     ER信号の状態
	 * @throws なし
	 */
	@Override
	public void onChangedERStatus(Serial serial, boolean er) {
		try {
			System.out.println("ERStatus change.");
			// ER信号とシリアルデバイスの状態を判定する
			if (!er && serial.isEnabled()) {
				// ER信号の状態がfalse且つシリアルデバイスが有効の状態
				System.out.println("ERStatus false set.");
				serial.setEnable(false);
			}
		} catch (ConnectionException e) {
			e.printStackTrace();
		}
	}
}







