import java.util.List;
import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.DeviceException;
import com.docomo_um.module.ExternalStatusException;
import com.docomo_um.module.IllegalSettingException;
import com.docomo_um.module.RegulationException;
import com.docomo_um.module.net.MessageBox;
import com.docomo_um.module.net.ShortMessage;
import com.docomo_um.module.net.ShortMessageBox;
import com.docomo_um.module.net.ShortMessageException;
import com.docomo_um.module.net.ShortMessageListener;
import com.docomo_um.module.net.ShortMessageManager;

/**
 * SMS通信のサンプルクラスです
 * <p>
 * SMSメッセージ受信リスナ設定、ショートメッセージ属性設定{@link #exec()}
 * SMS受信処理及び送信処理{@link #onSMSReceived(MessageBox,int,boolean)}
 * SMSメッセージの受信が拒否(メモリフル){@link #onSMSBoxFull(MessageBox)}
 * </p>
 * @see ShortMessageManager
 * @see ShortMessageBox
 * @see ShortMessage
 * @see MessageBox
 * @see ShortMessageListener
 *
 */
public class Main implements ShortMessageListener {
	/**受信ショートメッセージ番号*/
	private int recvNumber = 0;

	/**ショートメッセージマネジャーのインスタンス*/
	ShortMessageManager mngr = ShortMessageManager.getInstance();

	/**ショートメッセージボックスのインスタンス*/
	ShortMessageBox box = mngr.getShortMessageBox();

	/**
     * プログラムのエントリです
     * @param  なし
	 * @throws なし
     */
	public static void entry() {
		try {
			Thread.sleep(5000);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		System.out.println("entry start.");
		// インスタンスを作成して、SMS送受信処理をコールする
		new Main().exec();
	}

	/**
	 * SMSの送受信処理
	 * <P>
	 * SMS送受信ショートメッセージ属性設定{@link #sendMessagePropSet()}
	 * SMS送受信のリスナ設定{@link ShortMessageManager#setShortMessageListener(ShortMessageListener)}。
	 * </p>
	 * @param  なし
	 * @throws なし
	 */
	private void exec() {
		System.out.println("exec start.");
		try {
			// 送受信ショートメッセージ属性設定
			sendMessagePropSet();

			// 受信リスナ設定
			mngr.setShortMessageListener(this);
		} catch (DeviceException e) {
			e.printStackTrace();
		} catch (ShortMessageException e) {
			e.printStackTrace();
		} catch (ExternalStatusException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			System.out.println("process end");
		}
	}

	/**
	 * SMS情報受信処理
	 * <P>
	 * パラメータのmessageBoxは{@link ShortMessageManager#getShortMessageBox()}、
	 * または{@link ShortMessageManager#getSMSStatusReportBox()}で取得するインスタンスと
	 * 同一のインスタンスです
	 * パラメータのoverrideは受信ショートメッセージが他のショートメッセージを上書きして保存された場合はtrue、それ以外はfalse
	 * {@link ShortMessageBox#receive(ShortMessage)}には「false]固定を設定しました
	 * 未読のショートメッセージの取得処理{@link #getUnreadMsg()}
	 * 既読ショートメッセージの取得処理{@link #getReadMsg()}
	 * 番号指定メッセージの取得処理{@link #getMsgNumMsg()}
	 * ショートメッセージ属性の取得処理{@link #getMsgBoxProp()}
	 * ショートメッセージの送信処理{@link #sendMsg()}
	 * </p>
	 * @param  messageBox  受信ショートメッセージのインスタンス
	 * @param  messageNumber　受信ショートメッセージ番号
	 * @param  override  受信ショートメッセージ上書きフラグ
	 */
	@Override
	public void onSMSReceived(MessageBox messageBox, int messageNumber,boolean override) {
		System.out.println("onSMSReceived start.");
		System.out.println("onSMSReceived() called. type=["+messageBox.getType()+"], messageNumber=["+messageNumber+"], override=["+override+"]");
		recvNumber = messageNumber;
		try {
			// 未読ショートメッセージ取得
			getUnreadMsg();
			// 既読ショートメッセージ取得
			getReadMsg();
			// 番号指定ショートメッセージ取得
			getMsgNumMsg();
			// ショートメッセージ属性取得
			getMsgBoxProp();
			// ショートメッセージ送信処理
			sendMsg();
		} catch (DeviceException e) {
			e.printStackTrace();
		} catch (ExternalStatusException e) {
			e.printStackTrace();
		} catch (ShortMessageException e) {
			e.printStackTrace();
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (IllegalSettingException e) {
			e.printStackTrace();
		} catch (CommunicationFailureException e) {
			e.printStackTrace();
		} catch (RegulationException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
		System.out.println("onSMSReceived end.");
	}

	/**
	 * エリアメール受信処理
	 * <P>
	 * オーバーライドのみ、利用しない
	 * </p>
	 * @param  message
	 * @param  override
	 * @throws なし
	 */
	@Override
	public void onAreaMailReceived(ShortMessage message, boolean override) {
		recvNumber = message.getMessageNumber();
	}

	/**
	 * ショートメッセージの受信が拒否された場合、コールされる
	 * <P>
	 * SMS、または、SMSステータスレポートのショートメッセージを受信した際に、メッセージボックスに空きがなく、
	 * 且つ、保存されているショートメッセージがすべて未読({@link ShortMessage#STAT_UNREAD})である場合、
	 * ショートメッセージの受信が拒否され当該メソッドがコールされます
	 * ショートメッセージボックスのメッセージを削除する
	 * </p>
	 * @param  messageBox　ショートメッセージのインスタンス
	 * @throws なし
	 */
	@Override
	public void onSMSBoxFull(MessageBox messageBox) {
		System.out.println("onSMSBoxFull start.");
		System.out.println("onSMSBoxFull() called. type=["+messageBox.getType()+"]");
			try {
				// 受信メモリのメッセージ数を判定する
				if (box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_RECEIVE) >=
						box.getTotalMessageNumber(ShortMessageBox.ACTION_TYPE_RECEIVE)) {
					// 受信メモリのメッセージ数が最大値以上の場合
					System.out.println("onSMSBoxFull getTotalMessageNumber =" + box.getTotalMessageNumber(ShortMessageBox.ACTION_TYPE_RECEIVE));
					System.out.println("onSMSBoxFull before delete getSavedMessageNumber =" + box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_RECEIVE));
					// 受信メモリを削除する。
					box.delete(ShortMessageBox.DELETE_TYPE_READ_SENT_UNSENT);
					System.out.println("onSMSBoxFull after delete getSavedMessageNumber =" + box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_RECEIVE));
				}
			} catch (DeviceException e) {
				e.printStackTrace();
			} catch (ExternalStatusException e) {
				e.printStackTrace();
			} catch (ShortMessageException e) {
				e.printStackTrace();
			} catch (Exception e) {
				e.printStackTrace();
			}
		System.out.println("onSMSBoxFull end.");
	}

	/**
	 * 送信ショートメッセージ属性設定
	 * <P>
	 * システム利用のショートメッセージ形式設定
	 * センター番号及び国際アクセス番号設定
	 * 送信ショートメッセージ保存領域設定
	 * 受信ショートメッセージ保存領域設定
	 * </p>
	 * @param  なし
	 * @throws DeviceException
	 * @throws ShortMessageException
	 * @throws ExternalStatusException
	 */
	private void sendMessagePropSet()throws DeviceException,ShortMessageException,ExternalStatusException {
		System.out.println("sendMessagePropSet() start!");
		// ショートメッセージのフォーマットを設定する
		mngr.setMessageFormat(ShortMessageManager.FORMAT_TEXT);
		// SMSセンター番号と国際アクセスコードを設定する(環境により変更)
		mngr.setSMSCenterNumber("+81903101652", "145");
		// 送信メッセージはモジュール本体へ保存に設定する
		box.setStorage(ShortMessageBox.ACTION_TYPE_SEND,ShortMessageBox.STORAGE_TYPE_MODULE);
		// 受信メッセージはUIMへ保存に設定する
		box.setStorage(ShortMessageBox.ACTION_TYPE_RECEIVE,ShortMessageBox.STORAGE_TYPE_UIM);
		System.out.println("sendMessagePropSet() end !");
	}

	/**
	 * 未読ショートメッセージ取得
	 * <P>
	 * 未読ショートメッセージを取得して、ショートメッセージを出力する
	 * </p>
	 * @param  なし
	 * @throws DeviceException
	 * @throws ExternalStatusException
	 * @throws ShortMessageException
	 */
	private void getUnreadMsg() throws DeviceException,ExternalStatusException,ShortMessageException {
		System.out.println("get unread message start.");
		// 未読メッセージ情報を取得する
		List<ShortMessage> unredList = box.get(ShortMessage.STAT_UNREAD);
		// 未読メッセージ件数分繰り返し出力する
		for (ShortMessage msg : unredList) {
			printShortMessage(msg);
		}
		System.out.println("get unread message end.");
	}

	/**
	 * 既読ショートメッセージ取得
	 * <P>
	 * 既読ショートメッセージを取得して出力する
	 * </p>
	 * @param  なし
	 * @throws DeviceException
	 * @throws ExternalStatusException
	 * @throws ShortMessageException
	 */
	private void getReadMsg() throws DeviceException,ExternalStatusException,ShortMessageException {
		System.out.println("get read message start.");
		// 既読メッセージを取得する
		List<ShortMessage> redList = box.get(ShortMessage.STAT_READ);
		// 既読メッセージ件数分繰り返し出力する
		for (ShortMessage msg : redList) {
			printShortMessage(msg);
		}
		System.out.println("get read message end.");
	}

	/**
	 * 番号指定ショートメッセージ取得
	 * <P>
	 * ショートメッセージを番号指定で取得して出力する
	 * </p>
	 * @param  なし
	 * @throws DeviceException
	 * @throws ExternalStatusException
	 * @throws ShortMessageException
	 */
	private void getMsgNumMsg() throws DeviceException,ExternalStatusException,ShortMessageException {
		System.out.println("get number message start.");
		System.out.println("get number message. number=["+recvNumber+"]" + "start.");
		// 指定された番号のメッセージを取得する
		ShortMessage msg = box.get(recvNumber);
		// メッセージを出力する
		printShortMessage(msg);
		System.out.println("get number message end.");
	}

	/**
	 * ショートメッセージ情報取得
	 * <P>
	 * ショートメッセージ情報を取得して出力する
	 * </p>
	 * @param  なし
	 * @throws DeviceException
	 * @throws ExternalStatusException
	 * @throws ShortMessageException
	 */
	private void getMsgBoxProp() throws DeviceException,ExternalStatusException,ShortMessageException {
		System.out.println("get message box property start.");
		// 送信メモリに保存した送信ショートメッセージ数
		System.out.println("getSavedMessageNumber(SEND)=["+box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_SEND)+"]");
		// 受信メモリに保存した受信ショートメッセージ数
		System.out.println("getSavedMessageNumber(RECV)=["+box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_RECEIVE)+"]");
		// 送信データの保存場所を取得する
		System.out.println("getStorage(SEND)=["+box.getStorage(ShortMessageBox.ACTION_TYPE_SEND)+"]");
		// 受信データの保存場所を取得する
		System.out.println("getStorage(RECV)=["+box.getStorage(ShortMessageBox.ACTION_TYPE_RECEIVE)+"]");
		System.out.println("getTotalMessageNumber(SEND)=["+box.getTotalMessageNumber(ShortMessageBox.ACTION_TYPE_SEND)+"]");
		System.out.println("getTotalMessageNumber(RECV)=["+box.getTotalMessageNumber(ShortMessageBox.ACTION_TYPE_RECEIVE)+"]");
		System.out.println("get message box property end.");
	}

	/**
	 * ショートメッセージ送信処理
	 * <P>
	 * ショートメッセージ送信処理
	 * </p>
	 * @param  なし
	 * @throws IllegalArgumentException
	 * @throws ExternalStatusException
	 * @throws DeviceException
	 * @throws ShortMessageException
	 * @throws IllegalSettingException
	 * @throws CommunicationFailureException
	 * @throws RegulationException
	 */
	private void sendMsg() throws IllegalArgumentException,ExternalStatusException,DeviceException,ShortMessageException,IllegalSettingException,CommunicationFailureException, RegulationException {
		// ShortMessageのインスタンスを定義する
		ShortMessage sm = null;
		// ショートメッセージ番号の変数を定義する
		int number = 0;
		System.out.println("send message start.");
		// setShortMessageをコールし、送信ショートメッセージを作成する
		sm = setShortMessage("This is a test send message for sms!");
		// 送信済みメッセージ件数を判定する
		if (box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_SEND) >=
				box.getTotalMessageNumber(ShortMessageBox.ACTION_TYPE_SEND)){
			// 最大値以上の場合
			System.out.println("sendMsg getTotalMessageNumber =" + box.getTotalMessageNumber(ShortMessageBox.ACTION_TYPE_SEND));
			System.out.println("sendMsg before delete getSavedMessageNumber =" + box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_SEND));
			// 送信ショートメッセージの保存メモリを削除する
			box.delete(ShortMessageBox.DELETE_TYPE_READ_SENT_UNSENT);
			System.out.println("sendMsg after delete getSavedMessageNumber =" + box.getSavedMessageNumber(ShortMessageBox.ACTION_TYPE_SEND));
		}
		// 送信ショートメッセージをメモリに保存する
		number = box.save(sm);
		// ショートメッセージを送信する
		box.send(number, false, 0);
		System.out.println("send message end.");
	}

	/**
	 * ショートメッセージ情報出力
	 * <P>
	 * ショートメッセージ番号出力
	 * ショートメッセージ本体出力
	 * 送信元電話番号出力
	 * 送信元電話タイプ出力
	 * 送信先電話番号出力
	 * 送信先電話タイプ出力
	 * </p>
	 * @param  msg　ショートメッセージのインスタンス
	 * @throws なし
	 */
	private void printShortMessage(ShortMessage msg) {
		System.out.println("printShortMessage start.");
		// ショートメッセージ番号を取得し出力する
		System.out.println("number=["+msg.getMessageNumber()+"]");
		// ショートメッセージ情報を取得し出力する
		System.out.println("message=["+msg.getMessage()+"]");
		// ショートメッセージの送信元の電話番号を取得し出力する
		System.out.println("fromPhone=["+msg.getAttribute(ShortMessage.ATTR_FROM_PHONE_NUMBER)+"]");
		// 送信元の電話タイプを取得し出力する
		System.out.println("fromPhoneType=["+msg.getAttribute(ShortMessage.ATTR_FROM_PHONE_NUMBER_TYPE)+"]");
		// 送信先の電話番号を取得し出力する
		System.out.println("toPhone=["+msg.getToPhoneNumber()+"]");
		// 送信先の電話タイプを取得し出力する
		System.out.println("toPhoneType=["+msg.getToPhoneNumberType()+"]");
		System.out.println("printShortMessage end.");
	}

	/**
	 * 送信ショートメッセージ作成
	 * <P>
	 * 送信ショートメッセージ作成
	 * </p>
	 * @param  msg　ショートメッセージ本体
	 * @throws なし
	 */
	private ShortMessage setShortMessage(String msg) {
		System.out.println("setShortMessage start.");
		// ShortMessageのインスタンスを作成する
		ShortMessage sm = new ShortMessage();
		// ショートメッセージ情報をShortMessageのインスタンスに設定する
		sm.setMessage(msg);
		// 送信先の電話番号を設定する
		sm.setToPhoneNumber("00000000000");
		// 送信先の電話タイプを設定する
		sm.setToPhoneNumberType("129");
		System.out.println("setShortMessage end.");
		return sm;
	}
}
