import java.util.Calendar;
import java.util.Date;
import com.docomo_um.lang.Degree;
import com.docomo_um.module.ExternalStatusException;
import com.docomo_um.module.ModuleException;
import com.docomo_um.module.ModuleManager;
import com.docomo_um.module.RegulationException;
import com.docomo_um.module.location.Location;
import com.docomo_um.module.location.LocationException;
import com.docomo_um.module.location.LocationProvider;
import com.docomo_um.module.location.ProviderLocationListener;

/**
 * ロケーション情報取得のサンプルクラスです
 * <p>
 * ロケーション情報取得し、出力する{@link #exec()}
 * 現在位置提供を開始した時、測位状態通知{@link #onProvideLocationStarted(int)}
 * </p>
 * @see Calendar
 * @see Degree
 * @see ModuleManager
 * @see Location
 * @see LocationProvider
 * @see ProviderLocationListener
 * @see Date
 */
public class Main implements ProviderLocationListener {
	/**
     * プログラムのエントリです
     * @param  なし
	 * @throws なし
     */
	public static void entry() {
		// インスタンスを作成して、通信デバイスの情報設定処理をコールする
		new Main().exec();
	}

	/**
	 * 通信デバイスの情報設定処理
	 * <P>
	 * 測位方式 を取得する処理{@link LocationProvider#getAvailableLocationMethod()}
	 * ロケーション情報を取得する処理{@link LocationProvider#getLocation()}
	 * 取得された経度、緯度、高度、水平方向制度、情報取得時刻を出力する
	 * </p>
	 * @param  なし
	 * @throws なし
	 */
	private void exec() {
		try {
			while (true) {
				// 測位方式を保存するローカル変数を定義する
				int[] method = null;
				System.out.println("The measurement method get start.");
				// 測位方式が取得されるまで、処理を繰り返す
				while (method == null) {
					// スリープ(1秒)を行う
					try {
						Thread.sleep(1000);
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
					// 測位方式取得
					method = LocationProvider.getAvailableLocationMethod();
				}
				System.out.println("The measurement method get end.");
				// 測位機能提供クラスのインスタンス作成
				LocationProvider provider = null;
				System.out.println("Location provider get start.");
				for (int i = 0; i < method.length; i++){
					// 測位方式の種類を判定する
					if (method[i] == LocationProvider.METHOD_AUTO ||
						method[i] == LocationProvider.METHOD_STANDALONE ||
						method[i] == LocationProvider.METHOD_STANDARD) {
						// 測位機能提供クラスのインスタンスを取得する
						provider = ModuleManager.getInstance().getLocationProvider(method[i]);
						break;
					}
				}
				System.out.println("Location provider get end.");
				System.out.println("provider.getLocation start.");
				// Locationの変数を定義する
				Location location = null;
				if (provider != null) {
					// 受信リスナ設定
					provider.setProviderLocationStartedListener(this);
					// ロケーション情報を取得する
					location = provider.getLocation();
				} else {
					System.out.println("provider is null.");
					break;
				}
				System.out.println("provider.getLocation end.");
				// 経度取得
				System.out.println("Location information output Start.");
				// 経度を取得するため、Degreeのインスタンスを作成する
				Degree degree1 = new Degree(location.getLongitude().getFixedPointNumber());
				// 経度情報を出力する
				System.out.println("Longitude ：　" + degree1.getDegreePart()+ "(deg) "+
									degree1.getMinutePart() + "(min) " +
									degree1.getCentisecondPart()/100 + "(sec)");
				Thread.sleep(500);
				// 緯度取得
				// 緯度を取得するために、Degreeのインスタンスを作成する
				Degree degree2 = new Degree(location.getLatitude().getFixedPointNumber());
				// 緯度情報を出力する
				System.out.println("Latitude ：　" + degree2.getDegreePart()+ "(deg) "+
									degree2.getMinutePart() + "(min) " +
									degree2.getCentisecondPart()/100 + "(sec)");
				Thread.sleep(500);
				// 高度取得
				// 高度情報が取得可能か判定する
				if (location.getAltitude() != Location.ALTITUDE_UNKNOWN) {
					// 取得可能な場合、高度情報を出力する
					System.out.println("Altitude (m): "+ location.getAltitude());
				}
				// 水平方向精度取得
				// 水平方向精度情報が取得可能か判定する
				if (location.getAccuracy() != Location.ACCURACY_FINE &&
					location.getAccuracy() != Location.ACCURACY_COARSE &&
					location.getAccuracy() != Location.ACCURACY_NORMAL &&
					location.getAccuracy() != Location.ACCURACY_UNKNOWN) {
					// 取得可能な場合、水平方向精度情報を出力する
					System.out.println("Accuracy (m): "+location.getAccuracy());
				}
				Thread.sleep(500);
				// 位置情報取得の時刻
				// 位置情報取得の時刻を取得するため、Calendarのインスタンスを作成する
				Calendar calendar = Calendar.getInstance();
				// 位置情報取得の時刻(ミリ秒)をCalendarのインスタンスに設定する
				calendar.setTimeInMillis((long)location.getTimestamp());
				// 位置情報取得の時刻(年月日時分秒)を出力する
				System.out.println("get timestamp :" + calendar.toString());
				System.out.println("Location information output End.");
				Thread.sleep(60000);
			}
		} catch (ModuleException e) {
			e.printStackTrace();
		} catch (LocationException e) {
			e.printStackTrace();
		} catch (RegulationException e) {
			e.printStackTrace();
		} catch (ExternalStatusException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 *<p>
	 * 現在位置提供を開始した場合にコールされるメソッド
	 *</p>
	 *@param status 設定される測位状態
	 *@throws なし
	 */
	@Override
	public void onProvideLocationStarted(int status) {
		switch (status) {
			case LocationProvider.STATUS_IDLE:
			    // 現在位置未取得、緊急時の位置未提供状態
				System.out.println("Status is IDLE");
				break;
			case LocationProvider.STATUS_MEASUREMENT:
			    // 現在位置取得中状態
				System.out.println("Status is MEASUREMENT");
				break;
			case LocationProvider.STATUS_EMERGENCY  :
				// 緊急通報時の位置提供中状態
				System.out.println("Status is EMERGENCY");
				break;
			    // その他の状態
			default :
				System.out.println("Status is " + status);
		}
	}
}

