import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;
import com.docomo_um.module.CommunicationFailureException;
import com.docomo_um.module.DeviceException;
import com.docomo_um.module.ExternalStatusException;
import com.docomo_um.module.IllegalSettingException;
import com.docomo_um.module.ModuleException;
import com.docomo_um.module.ModuleManager;
import com.docomo_um.module.RegulationException;
import com.docomo_um.module.net.IPSettingInfo;
import com.docomo_um.module.net.PacketController;
import com.docomo_um.module.net.PacketSession;
import com.docomo_um.module.net.Session;
import com.docomo_um.module.net.SessionException;

/**
 * ソケット(TCP/IP)通信のサンプルクラスです
 * <p>
 * パケット通信出来る状況を確認する{@link #exec()}
 * サーバソケットを作成する{@link #exec()}
 * サーバソケットの受信処理{@link #SocketRead(Socket)}
 * クライアントソケット送信処理{@link #SocketClient()}
 * </p>
 * @see ModuleManager
 * @see IPSettingInfo
 * @see PacketController
 * @see PacketSession
 * @see Session
 */
public class Main {
	/**ソケットポート定義*/
	private static final int SOCKET_PORT = 5550;
	/**サーバのホスト名定義*/
	private static final String HOSTNAME = "sampleserver";

	/**
     * プログラムのエントリです
     * @param  なし
	 * @throws なし
     */
	public static void entry() {
		System.out.println("entry start.");
		// インスタンスを作成して、ソケット受信処理をコールする
		new Main().exec();
		System.out.println("entry end.");
	}
	/**
	 * ソケット受信処理
	 * <P>
	 * IP情報インスタンスを作成する{@link IPSettingInfo#IPSettingInfo(int, String, String, String)}
	 * パケット通信を設定して{@link PacketSession#getPDPTypeInfo()}
	 * パケット通信出来る状況を確認する{@link PacketSession#send()}
	 * ポートを指定してサーバソケットのインスタンスを作成する{@link ServerSocket#ServerSocket(int)}
	 * 受信ソケットを作成する{@link ServerSocket#accept()}、クライアントから接続待ち
	 * </p>
	 * @param  なし
	 * @throws なし
	 */
	private void exec() {
		// ServerSocketのインスタンスを定義する
		ServerSocket serverSocket = null;
		// IP情報作成
		// IPSettingInfoのインスタンスを作成し、認証情報を作成する
		IPSettingInfo settingInfo = new IPSettingInfo(IPSettingInfo.IP_AUTHENTICATION_NONE,null,null);
		// IPアドレスの割り当て方式を設定する
		settingInfo.setAllocatingIP(IPSettingInfo.ALLOCATING_IP_STATIC);
		System.out.println("The connecting of Socket server is start.");
		try {
			//パケット通信確認情報を作成する
			PacketController packetController = ModuleManager.getInstance().getPacketController();
			PacketSession session = (PacketSession) packetController.createSession("i-mode.docomo.ne.jp");
			// セッションの接続待ち
			Thread.sleep(3000);
			// パケット通信のPDPType情報を設定する
			session.setPDPTypeInfo(settingInfo);
			// セッションの接続待ち
			Thread.sleep(3000);
			System.out.println("session.send() Start.");
			// セッション情報を送信する
			session.send();
			System.out.println("session.send() End.");
			// セッションの接続状態を判断する
			if (session.getConnectedStatus() ==Session.CONNECTION_STATUS_OUTGOING_CALL ||
				session.getConnectedStatus() == Session.CONNECTION_STATUS_INCOMING_CALL ||
				session.getConnectedStatus() == Session.CONNECTION_STATUS_CONNECT) {
				// セッション接続中を設定する
				System.out.println("Session is connecting.");
			} else {
				// セッション未接続を出力する
				System.out.println("Session is not connecting.");
				return;
			}
			while (true) {
				//サーバソケットのインスタンスを作成する
				serverSocket = new ServerSocket(SOCKET_PORT);
				//ソケット受信待ち
				System.out.println("The socket accept is starting.");
				// サーバソケットを受信待ちにする
				Socket socket = serverSocket.accept();
				System.out.println("The socket accept is end.");
				System.out.println("The SocketRead is start.");
				// スレッド待ち処理
				Thread.sleep(3000);
				// 受信処理を呼び出す
				SocketRead(socket);
				System.out.println("The SocketRead is end.");
				System.out.println("The SocketClient is start.");
				// 送信処理を呼び出す
				SocketClient();
				System.out.println("The SocketClient is end.");
				System.out.println("ServerSocket close start.");
				try {
					//サーバソケットをクローズする
					serverSocket.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
				System.out.println("ServerSocket close end.");
			}
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ModuleException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		} catch (SessionException e) {
			e.printStackTrace();
		} catch (CommunicationFailureException e) {
			e.printStackTrace();
		} catch (RegulationException e) {
			e.printStackTrace();
		} catch (IllegalSettingException e) {
			e.printStackTrace();
		} catch (ExternalStatusException e) {
			e.printStackTrace();
		} catch (DeviceException e) {
			e.printStackTrace();
		} catch (NullPointerException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (serverSocket != null) {
				System.out.println("ServerSocket close start.");
				try {
					//サーバソケットをクローズする
					serverSocket.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
				System.out.println("ServerSocket close end.");
			}
		}
	}

	/**
	 * ソケット送信のクライアント処理
	 * <P>
	 * サーバソケットに接続するクライアントソケットを作成する{@link Socket#Socket(String, int)}
	 * サーバソケットへ送信する{@link OutputStream#write(byte[])}
	 * </p>
	 * @param  なし
	 * @throws なし
	 */
	private void  SocketClient() {
		// Socketのインスタンスを定義する
		Socket clientSocket = null;
		try {
			// ホスト名とポートを指定して、Socketのインスタンスを作成する
			clientSocket = new Socket(HOSTNAME, SOCKET_PORT);
			System.out.println("SocketClient SoceketSend start.");
			// 送信データを作成する
			for (int j = 0; j < 10; j++) {
				byte[] buf = new byte[10];
				for (int i = 0; i < buf.length; i++) {
					// 送信データを配列に設定する
					buf[i] = (byte) ((byte)'m' + (byte)j);
				}
				// OutputStreamのインスタンスを作成する
				OutputStream os = clientSocket.getOutputStream();
				// 送信処理
				os.write(buf);
			}
		} catch (UnknownHostException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			// Socketインスタンスを判定する
			if (clientSocket != null) {
				// null以外の場合
				System.out.println("SocketClient SocketClose start.");
				try {
					// Socketのインスタンスをクローズする
					clientSocket.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
				System.out.println("SocketClient SocketClose end.");
			}
		}
	}

	/**
	 * ソケット受信処理
	 * <P>
	 * 受信ソケットより、入力ストリームを作成する{@link Socket#getInputStream()}
	 * 入力ストリームより、受信データを取得する{@link InputStream#read(byte[])}
	 * </p>
	 * @param  socket 受信情報を格納するソケットインスタンス
	 * @throws なし
	 */
	private void  SocketRead(Socket socket) {
		// 受信データのバイト配列を作成する
		byte[] buf = new byte[1024];
		try {
			System.out.println("SocketRead SocketReceive start.");
			// InputStreamのインスタンスを作成する
			InputStream is = socket.getInputStream();
			// 受信データを取得する
			is.read(buf);
			// ループで受信データを出力する
			for (int i = 0; i < buf.length; i++) {
				// 受信終了を判断する
				if (buf[i] != 0) {
					// 終了以外の場合、受信メッセージを出力する
					System.out.println("buf[" + i + "]=" +buf[i]);
				} else {
					// 終了の場合、終了メッセージを出力する
					System.out.println("SocketRead break.");
					// 受信データ出力処理ループを抜ける
					break;
				}
			}
			System.out.println("SocketRead SocketReceive end.");
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			// 受信Socketのインスタンスを判定する
			if (socket != null) {
				// null以外の場合
				System.out.println("SocketRead socket close start.");
				try {
					// ソケットをクローズする
					socket.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
				System.out.println("SocketRead socket close end.");
			}
		}
	}
}

